<?php

namespace admission\models;

use Yii;
use yii\base\Model;

/**
 * Registration model
 *
 * @property string $thai_id
 */
class Registration extends Model
{
    public $thai_id;

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            ['thai_id', 'required', 'message' => 'กรุณากรอกเลขบัตรประชาชน'],
            ['thai_id', 'string', 'length' => 13, 'message' => 'เลขบัตรประชาชนต้องมี 13 หลัก'],
            ['thai_id', 'match', 'pattern' => '/^[0-9]{13}$/', 'message' => 'เลขบัตรประชาชนต้องเป็นตัวเลขเท่านั้น'],
            ['thai_id', 'validateThaiID'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'thai_id' => 'เลขบัตรประชาชน (Thai ID Number)',
        ];
    }

    /**
     * Validates the Thai ID number using check digit algorithm
     *
     * @param string $attribute the attribute currently being validated
     * @param array $params the additional name-value pairs given in the rule
     */
    public function validateThaiID($attribute, $params)
    {
        if (!$this->hasErrors()) {
            $id = $this->$attribute;
            
            if (strlen($id) !== 13 || !ctype_digit($id)) {
                $this->addError($attribute, 'เลขบัตรประชาชนไม่ถูกต้อง');
                return;
            }
            
            // Calculate check digit using Thai ID algorithm
            $sum = 0;
            for ($i = 0; $i < 12; $i++) {
                $sum += (int)$id[$i] * (13 - $i);
            }
            
            $mod = $sum % 11;
            $checkDigit = (11 - $mod) % 10;
            
            if ($checkDigit != (int)$id[12]) {
                $this->addError($attribute, 'เลขบัตรประชาชนไม่ถูกต้อง กรุณาตรวจสอบอีกครั้ง');
            }
        }
    }

    /**
     * Process the registration
     *
     * @return bool whether the registration was successful
     */
    public function register()
    {
        if ($this->validate()) {
            // Add your registration logic here
            // For example: save to database, send email, etc.
            
            Yii::info('Registration attempt for Thai ID: ' . $this->thai_id, __METHOD__);
            
            // Return true if registration is successful
            return true;
        }
        
        return false;
    }
}
